%{
/* scanner.l
   
   Lexical analyzer for definition of Network Expect PDUs

   Copyright (C) 2007, 2008, 2009, 2010 Eloy Paris

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "pbuild-priv.h"
#include "parser.h"

%}

%option reentrant
%option bison-bridge
%option noyywrap
/*
 * We don't use unput, so don't generate code for it.
 */
%option nounput

INTEGER		[0-9]+|0[xX][0-9A-Fa-f]+
REALNUM		-?([0-9]+)?\.[0-9]+
IPSTRING	[0-9]+\.[0-9]+\.[0-9]+\.[0-9]+
HOSTNAME	[0-9A-Za-z\-]+\.([0-9A-Za-z\-]+\.)*[0-9A-Za-z\-]+

%x IN_COMMENT
%%

		/* C-style comments. See "How can I match C-style comments?"
		   in the flex FAQ */
<INITIAL>{
"/*"		BEGIN(IN_COMMENT);
}
<IN_COMMENT>{
"*/"		BEGIN(INITIAL);
[^*\n]+		/* eat comment in chunks */
"*"		/* eat the lone star */
\n		/* eat end of line */
}

		/* String enclosed with double quotes */
\"[^\"]*\"	{
		    yytext[yyleng - 1] = '\0';
		    yyleng = strlen(yytext + 1);
		    yylval->string = xstrdup(yytext + 1);
		    return STRING;
		}

		/* String enclosed with single quotes */
\'[^\']*\'	{
		    yytext[yyleng - 1] = '\0';
		    yyleng = strlen(yytext + 1);
		    yylval->string = xstrdup(yytext + 1);
		    return STRING;
		}

		/* Makes it possible to specify a range without having to
		   enclose everything with quotes. */
\<[^<>]*>	{
		    yylval->string = xstrdup(yytext);
		    return STRING;
		}

"("		{ return '('; }
")"		{ return ')'; }
"/"		{ return '/'; }
","		{ return ','; }
"="		{ return '='; }

"no"		{ return NO; }

		/* A symbol. Note that the dash ('-') is allowed to be
		   part of a symbol */
[A-Za-z][A-Za-z0-9_\-]*	{
		    yylval->string = xstrdup(yytext);
		    return NAME;
		}

{INTEGER}	{
		    yylval->i = strtoul(yytext, NULL, 0);
		    return INTEGER;
		}

{REALNUM}	{
		    yylval->d = strtod(yytext, NULL);
		    return DOUBLE;
		}

{IPSTRING}	{
		    struct addr a;

		    addr_pton(yytext, &a);
		    yylval->ipaddr = a.addr_ip;
		    return IPADDR;
		}

[^=;,()/ \t\r\n]+   {
		    yylval->string = xstrdup(yytext);
		    return STRING;
                }

[ \t\r\n]+	/* eat whitespace */

%%
