#ifndef _NEXP_SPEAKERS_H
#define _NEXP_SPEAKERS_H

/* nexp_speakers.h
  
   Definitions for Network Expect speakers

   Copyright (C) 2007, 2008, 2009 Eloy Paris

   This is part of Network Expect.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
    
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
    
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <dnet.h>
#include <pcap.h>

#define NEXP_SPEAKER_NAMELEN 40

#define DEFAULT_IPV4_SPEAKER_NAME "ip"
#ifdef __linux__
#define DEFAULT_IPV6_SPEAKER_NAME "ip6"
#endif
#define DEFAULT_HEX_SPEAKER_NAME "hex"
#define DEFAULT_STDOUT_SPEAKER_NAME "stdout"
#define DEFAULT_NULL_SPEAKER_NAME "null"

#define SPEAKER_SPAWN_ID_VARNAME "speaker_id"

/*
 * Better to not make it part of the enum below to prevent compiler warnings.
 * Used only during processing of the spawn_network command. Obviously
 * an arbitrary number.
 */
#define SPEAKER_NONE 0xffff1073

enum speaker_types {
    SPEAKER_ETHER,
    SPEAKER_IPV4,
    SPEAKER_IPV4USR,
#ifdef __linux__
    SPEAKER_IPV6,
#endif
    SPEAKER_HEX,
    SPEAKER_PCAP,
    SPEAKER_STDOUT,
    SPEAKER_NULL,
    SPEAKER_STREAM,
    SPEAKER_DGRAM
};

/*
 * The nexp_speaker structure keeps track of a speaker that has been
 * activated via the spawn_network command. Speakers are specified using
 * the -o switch of the send_network command.
 */
struct nexp_speaker {
    struct nexp_speaker *next;

    char name[NEXP_SPEAKER_NAMELEN + 1];

    struct timeval ts; /* Time when a packet was last sent via this speaker */

    enum speaker_types type;

    union {
	/* Used when sending packets to a PCAP file */
	struct {
	    pcap_t *pd;
	    pcap_dumper_t *pdumper;
	    const char *fname;
	} __pcap;

	/* Used when sending IPv4 packets routed in userspace by libusr */
	struct {
	    struct iftable *ifaces;
	    struct rtable *routes;
	    struct neightable *neighbors;
	} __ip4usr;

	/* Used when sending IPv4 packets routed by the kernel */
	ip_t *__ip;

#ifdef __linux__
	/* Used when sending IPv6 packets routed by the kernel */
	int __ip6;
#endif

	/* Used when sending Ethernet frames */
	struct {
	    eth_t *ether;
	    const char *ifname;
	} __iface;

	/* Used when reading from a TCP or UDP socket */
	struct {
	    int fd;
	    char *dsthost;
	    char *dstport;
	} __socket;
    } __type_u;
};

#define _ip4usr __type_u.__ip4usr
#define _ip __type_u.__ip
#ifdef __linux__
#define _ip6 __type_u.__ip6
#endif
#define _iface __type_u.__iface
#define _pcap __type_u.__pcap
#define _socket __type_u.__socket

extern struct nexp_speaker *lookup_speaker(const char *);
extern void speakers_info(void);
extern void close_speaker(struct nexp_speaker *);
extern void close_speakers(void);
extern void create_default_speakers(void);
extern struct nexp_speaker *nexp_newspeaker(struct nexp_speaker *, char *);

#endif /* _NEXP_SPEAKERS_H */
