/*
    Copyright (C) 2006 Tobias Koenig <tobias.koenig@credativ.de>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
*/

#ifndef QLDAP_H
#define QLDAP_H

#include <QtCore/QMap>
#include <QtCore/QObject>
#include <QtCore/QStringList>

class QLdapEntry;
class QLdapResponse;

class QLdap : public QObject
{
  Q_OBJECT

  public:

    /**
     * The scope for LDAP searches
     *  - Base Searches on the same level as the base DN
     *  - OneLevel Searches one level below the base DN
     *  - Sub Searches on the whole subtree of the base DN
     */
    enum Scope
    {
      Base,
      OneLevel,
      Sub
    };

    /**
     * Constructs a QLdap object.
     */
    QLdap( QObject *parent = 0 );

    /**
     * Constructs a QLdap object with a host name and port number associated.
     */
    QLdap( const QString &host, quint16 port = 389, QObject *parent = 0 );

    /**
     * Destroys the QLdap object.
     */
    virtual ~QLdap();

    /**
     * Sets the host name.
     */
    void setHost( const QString &hostName );

    /**
     * Returns the host name.
     */
    QString host() const;

    /**
     * Sets the port number.
     */
    void setPort( int port );

    /**
     * Returns the port number.
     */
    int port() const;

    /**
     * Sets the user name.
     */
    void setUser( const QString &user );

    /**
     * Returns the user name.
     */
    QString user() const;

    /**
     * Sets the password.
     */
    void setPassword( const QString &password );

    /**
     * Returns the password.
     */
    QString password() const;

    /**
     * Sets the base DN
     */
    void setBaseDn( const QString &baseDn );

    /**
     * Returns the base DN.
     */
    QString baseDn() const;

    /**
     * Searches for all entries with base DN @param base in the @param scope, which
     * match the given @param filter.
     * If @param attributes is not empty, only the contained attributes are returned.
     * @param sortBy is the name of the field which is used for sorting the return.
     *
     * @return A QLdapResponse filled with the matching entries on success, an invalid
     *         QLdapResponse otherwise.
     */
    QLdapResponse search( const QString &base, Scope scope, const QString &filter,
                          const QStringList &attributes = QStringList(),
                          const QString &sortBy = QString() );

    /**
     * Adds the @param entry under the given @param dn to the LDAP directory.
     *
     * @return true on success, false otherwise.
     */
    bool add( const QString &dn, const QLdapEntry &entry );

    /**
     * Adds the set attributes of @param entry to the LDAP entry with the
     * given @param dn.
     *
     * @return true on success, false otherwise.
     */
    bool addAttributes( const QString &dn, const QLdapEntry &entry );

    /**
     * Modifies the set attributes of @param entry in the LDAP entry with the
     * given @param dn.
     *
     * @return true on success, false otherwise.
     */
    bool modifyAttributes( const QString &dn, const QLdapEntry &entry );

    /**
     * Removes the set attributes of @param entry in the LDAP entry with the
     * given @param dn.
     *
     * @return true on success, false otherwise.
     */
    bool removeAttributes( const QString &dn, const QLdapEntry &entry );

    /**
     * Removes the entry with the given @param dn from the LDAP directory.
     *
     * @return true in success, false otherwise.
     */
    bool remove( const QString &dn );

    /**
     * Renames the entry with the given @param dn to the entry with the
     * given @param newDn.
     *
     * @param baseDn The parent base dn.
     * @param deleteOld If true the @param dn is deleted automatically.
     */
    bool rename( const QString &dn, const QString &newDn, const QString &baseDn, bool deleteOld );

    /**
     * Returns the string presentation of the last occured error.
     */
    QString errorString() const;

  private:
    bool change( const QString&, const QLdapEntry&, int );

    bool connect();
    bool bind();
    bool unbind();
    bool disconnect();

    class Private;
    Private *d;
};

class QLdapEntry
{
  public:
    typedef QList<QLdapEntry> List;

    QLdapEntry();
    ~QLdapEntry();

    QString dn() const;
    void setDn( const QString &dn );

    QStringList attributeNames() const;

    QString value( const QString &attributeName ) const;
    QStringList values( const QString &attributeName ) const;

    void setValue( const QString &attributeName, const QString &value );
    void addValue( const QString &attributeName, const QString &value );

    void clearValue( const QString &attributeName );

    void dump() const;

  private:
    QString mDn;
    QMap<QString, QStringList> mValues;
};


class QLdapResponse
{
  public:
    QLdapResponse();
    ~QLdapResponse();

    bool isValid() const;

    QLdapEntry::List entries() const;
    void setEntries( const QLdapEntry::List &entries );

    void dump() const;

  private:
    QLdapEntry::List mEntries;
    bool mValid;
};

#endif
